// $Id: CClip.cpp,v 1.3 2007/02/08 21:07:54 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CClip.hpp"
#include <Exceptions/CException.hpp>
using Exponent::GUI::Windowing::CClip;
using Exponent::Exceptions::CException;

//	===========================================================================
CClip::CClip()
	 : m_isClipped(false)
	 , m_image(NULL)
	 , m_initialised(false)
#ifndef WIN32
	 , m_theContext(NULL)
#endif
{
	// NULL the native image
	NULL_POINTER(m_image);

	// We are not setup and not clipped
	m_initialised = false;
	m_isClipped   = false;

	// Null quartz context
#ifndef WIN32
	NULL_POINTER(m_theContext);
#endif
}

//	===========================================================================
CClip::~CClip()
{
	this->uninitialise();
}

//	===========================================================================
void CClip::initialise(CNativeImage *image, const CRect &area, void *context)
{
	// Check we have valid parameteres
#ifdef WIN32
	if (image == NULL)
	{
		throw CException("Native image is NULL", "CClip::intialised(CNativeImage *, const CRect &, void *)");
	}
#else
	if (context == NULL)
	{
		throw CException("Context is NULL", "CClip::intialised(CNativeImage *, const CRect &, void *)");
	}
	m_theContext = (CGContextRef)context;
#endif

	// Store the other values
	m_image		  = image;
	m_area		  = area;
	m_currentClip = area;
	m_initialised = true;
}

//	===========================================================================
void CClip::uninitialise()
{
	// Clear the current clipping area
	this->clearClipRegion();

	// Forget about the image
	NULL_POINTER(m_image);

	// Not initialised
	m_initialised = false;
	m_isClipped   = false;

	// Forget about the quartz context
#ifndef WIN32
	NULL_POINTER(m_theContext);
#endif
}

//	===========================================================================
void CClip::getCurrentClipRegion(CRect &area)
{
#ifdef WIN32
	// Check we are intialised
	if (m_image && m_initialised)
	{
		// Get the clipping area
		RECT theRect;
		GetClipBox(m_image->getGraphicsHandle()->m_drawContext, &theRect);

		// Store it in the area
		area.setFromRect(theRect);
	}
	area = m_currentClip;
#else
	area = m_currentClip;
#endif
}

//	===========================================================================
void CClip::setCurrentClipRegion(const CRect &area)
{
	// Store the clipping area
	m_currentClip = area;

#ifdef DEBUG_CLIPPING
	return;
#endif

#ifdef WIN32
	// Check we are intialised
	if (m_image && m_initialised)
	{
		// Create a windows region
		HRGN region = CreateRectRgn(area.getLeft(), area.getTop(), area.getRight(), area.getBottom());

		// Select it as the new clipping area
		SelectClipRgn(m_image->getGraphicsHandle()->m_drawContext, region);

		// We dont need the region any more
		DeleteObject(region);

		// We are now clipped
		m_isClipped = true;
	}
#else

	// Store the clip rect, inverted for quarts
	CGRect clipRect;
	clipRect.origin.x	 = m_currentClip.getLeft() - 0.5f;
	clipRect.origin.y	 = m_area.getHeight() - (m_currentClip.getTop() + m_currentClip.getHeight()) - 0.5f;
	clipRect.size.width  = m_currentClip.getWidth() + 1.5f;
	clipRect.size.height = m_currentClip.getHeight() + 1.5f;
	
	// Apply the clipping
	CGContextRestoreGState(m_theContext);
	CGContextSaveGState(m_theContext);
	CGContextClipToRect(m_theContext, clipRect);
#endif
}

//	===========================================================================
void CClip::clearClipRegion()
{
#ifdef WIN32
	// Check we have everything and we are actually clipped
	if (m_image && m_isClipped && m_initialised)
	{
		// Check we have a valid draw context
	 	if (m_image->getGraphicsHandle()->m_drawContext)
	 	{
			// Restore the full window clipping area
			RestoreDC(m_image->getGraphicsHandle()->m_drawContext, -1);
		}
	}
#else
	// Restore the clipped rectangle
	this->setCurrentClipRegion(CRect(0, 0, m_area.getWidth(), m_area.getHeight()));	
#endif
}
